//
//  HomeHeader.swift
//  Do It
//
//  Created by Jim Dovey on 2/11/20.
//  Copyright © 2020 Jim Dovey. All rights reserved.
//

import SwiftUI

struct HomeHeader: View {
    // START:DataLayout
    static let layout: [[TodoItemGroup]] = [
        [.today, .scheduled],
        [.all, .overdue],
    ]
    // END:DataLayout

    // START:HeaderBody
    var body: some View {
        VStack {
            ForEach(Self.layout, id: \.self) { row in
                HStack(spacing: 12) {
                    ForEach(row, id: \.self) { group in
                        // START:HeaderContent
                        NavigationLink(destination: TodoList(group: group)) {
                            // END:HeaderContent
                            // <literal:elide> ... </literal:elide>
                            // START:HeaderContent
                            // END:HeaderBody
                            // START_HIGHLIGHT
                            HeaderItem(group: group)
                            // END_HIGHLIGHT
                            // START:HeaderBody
                        }
                        // END:HeaderContent
                    }
                }
            }
        }
    }
    // END:HeaderBody

    // START:HeaderItem
    private struct HeaderItem: View {
        let group: TodoItemGroup
        @State var itemCount: Int = 0

        // END:HeaderItem
        // START:DynamicContent
        // START_HIGHLIGHT
        @EnvironmentObject var data: DataCenter
        // END_HIGHLIGHT

        // START:HeaderItem
        var body: some View {
            VStack(alignment: .leading) {
                // END:HeaderItem
                // <literal:elide> ... </literal:elide>
                // END:DynamicContent
                // START:HeaderItem
                HStack {
                    group.icon
                    Spacer()
                    Text("\(itemCount)")
                        .foregroundColor(.primary)
                        .font(.system(.title, design: .rounded))
                        .fontWeight(.bold)
                }
                
                Text(group.title)
                    .foregroundColor(.secondary)
                    .font(.system(.subheadline, design: .rounded))
                    .fontWeight(.medium)
                // START:DynamicContent
            }
            // END:HeaderItem
            // START_HIGHLIGHT
            .onReceive(data.$todoItems) { _ in
                self.itemCount = self.group.items(from: self.data).count
            }
            // END_HIGHLIGHT
            // <literal:elide> ... </literal:elide>
            // END:DynamicContent
            // START:HeaderItem
            .padding()
            .background(
                RoundedRectangle(cornerRadius: 15, style: .continuous)
                    .fill(Color(.tertiarySystemBackground)) // <label id="code.ch6.tertiaryBackground" />
            )
            // START:DynamicContent
        }
    }
    // END:HeaderItem
    // END:DynamicContent
}

struct HomeHeader_Previews: PreviewProvider {
    static var previews: some View {
        // START:Preview
        ForEach(ColorScheme.allCases, id: \.self) { colorScheme in
            HomeHeader()
                .padding()
                .background(Color(.systemGroupedBackground)) // <label id="code.ch6.preview.background" />
                .colorScheme(colorScheme)
        }
        .previewLayout(.sizeThatFits)
        .environmentObject(DataCenter())
        // END:Preview
    }
}
